/*
 * Solar2.cxx
 *
 * Program to demonstrate how to use a local
 * coordinate method to position parts of a 
 * model in relation to other model parts.
 *
 * Draws a simple solar system, with a sun, planet and moon.
 * Based on sample code from the OpenGL programming guide
 *		by Woo, Neider, Davis.  Addison-Wesley.
 *
 * Author: Samuel R. Buss
 *
 * Software accompanying the book
 *		3D Computer Graphics: A Mathematical Introduction with OpenGL,
 *		by S. Buss, Cambridge University Press, 2003.
 *
 * Software is "as-is" and carries no warranty.  It may be used without
 *   restriction, but if you modify it, please change the filenames to
 *   prevent confusion between different versions.
 * Bug reports: Sam Buss, sbuss@ucsd.edu.
 * Web page: http://math.ucsd.edu/~sbuss/MathCG
 *
 * USAGE:
 *    Press "r" key to toggle (off and on) running the animation
 *    Press "s" key to single-step animation
 *    The up arrow key and down array key control the
 *			time step used in the animation rate.  Each key
 *			press multiplies or divides the times by a factor 
 *			of two (2).
 *	  Press ESCAPE to exit.
 *
 * 2011-11-25 - Some minor modifications
 * Geoff R. McLane - reports-at-geoffair-dot-info
 *
 */
#ifdef _MSC_VER
#pragma warning ( disable : 4305 ) // truncation from 'double' to 'GLfloat'
#endif // _MSC_VER

#include "Solar2.hxx"   
#include <stdio.h>  // for printf()
#include <iostream>
#include <sstream>
#include <string>
#include <iomanip>
#include <cstdlib>
#include <GL/glut.h>	// OpenGL Graphics Utility Library
#include "Timer.hxx"

using namespace std;

static GLenum spinMode = GL_TRUE;
static GLenum singleStep = GL_FALSE;

// These three variables control the animation's state and speed.
static float HourOfDay = 0.0;
static float DayOfYear = 0.0;
//static float AnimateIncrement = 24.0;  // Time step for animation (hours)
static float AnimateIncrement = 2.0;  // Time step for animation (hours)

static float cameraDistance = -8.0;
static float cameraAngleX = 15.0;  // pitch
static float cameraAngleY = 0;     // heading

// global variables
void *font = GLUT_BITMAP_8_BY_13;
int screenWidth = 600;
int screenHeight = 360;

bool mouseLeftDown = false;
bool mouseRightDown = false;
float mouseX, mouseY;

// forward references
void showFPS();



static void Key_Help(void)
{
    printf("Key HELP\n");
    printf(" ESC  = Exit application.\n");
    printf(" r    = Restart (or toggle) animation. ");
    printf("(spinMode %s)\n", (spinMode ? "On" : "Off"));
    printf(" R    = Reset original values.\n");
    printf(" s    = Set single step.\n");

    printf(" Up   = Increment annimation time step (%f)\n", AnimateIncrement);
    printf(" Down = Decrement annimation time step\n");
    printf(" ?/h  = This HELP output.\n");
}

// Set Reset initial values
void Key_R()
{
    singleStep = GL_FALSE;   // Not single step
	spinMode   = GL_TRUE;	 // Restart animation
    AnimateIncrement = 2.0;  // Time step for animation (hours)
    cameraDistance = -8.0;   // back away a bit
    cameraAngleX = 15.0;     // pitch
    cameraAngleY = 0;        // heading
}

// glutKeyboardFunc is called below to set this function to handle
//		all normal key presses.  
static void KeyPressFunc( unsigned char Key, int x, int y )
{
	switch ( Key ) {
	case 'R':
        Key_R();
        break;
	case 'r':
		Key_r();
		break;
	case 's':
	case 'S':
		Key_s();
		break;
    case '?':
    case 'h':
    case 'H':
        Key_Help();
        break;
	case 27:	// Escape key
		exit(0);
	}
}

// glutSpecialFunc is called below to set this function to handle
//		all special key presses.  See glut.h for the names of
//		special keys.
static void SpecialKeyFunc( int Key, int x, int y )
{
	switch ( Key ) {
	case GLUT_KEY_UP:		
		Key_up();
		break;
	case GLUT_KEY_DOWN:
		Key_down();
		break;
	}
}


static void Key_r(void)
{
	if ( singleStep ) {			// If ending single step mode
		singleStep = GL_FALSE;
		spinMode = GL_TRUE;		// Restart animation
        printf("r key - Restart animation\n");
	}
	else {
		spinMode = !spinMode;	// Toggle animation on and off.
        printf("r key - Toggle animation to %s\n", (spinMode ? "On" : "Off"));
	}
}

static void Key_s(void)
{
	singleStep = GL_TRUE;
	spinMode = GL_TRUE;
    printf("s key - Set single step.\n");
}

static void Key_up(void)
{
    AnimateIncrement *= 2.0;			// Double the animation time step
    printf("Increment annimation time step to %f\n", AnimateIncrement);
}

static void Key_down(void)
{
    AnimateIncrement /= 2.0;			// Halve the animation time step
    printf("Decrease annimation time step to %f\n", AnimateIncrement);
	
}

float axisLength = 2.5f;
float headLength = axisLength - 0.25f;
float headWidth = 0.05f;

void XYZaxes()
{
	glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT | GL_LINE_BIT );
	glLineWidth( 3.0f );
	glDisable( GL_LIGHTING );
	glBegin( GL_LINES );
		/* color X axis red */
		glColor3f(  1.0f, 0.0f, 0.0f );
		glVertex3f( 0.0f, 0.0f, 0.0f );
		glVertex3f( axisLength, 0.0f, 0.0f );

		/* color Y axis green */
		glColor3f(  0.0f, 1.0f, 0.0f );
		glVertex3f( 0.0f, 0.0f, 0.0f );
		glVertex3f( 0.0f, axisLength, 0.0f );

		/* color Z axis blue */
		glColor3f(  0.0f, 0.0f, 1.0f );
		glVertex3f( 0.0f, 0.0f, 0.0f );
		glVertex3f( 0.0f, 0.0f, axisLength );
	glEnd();

	glBegin( GL_TRIANGLES );
		glColor3f(  1.0f, 0.0f, 0.0f ); // red
		glVertex3f( axisLength, 0.0f, 0.0f );
		glVertex3f( headLength, -headWidth, 0.0f );
		glVertex3f( headLength, headWidth, 0.0f );
	
		glColor3f(  0.0f, 1.0f, 0.0f ); // green
		glVertex3f( 0.0f, axisLength, 0.0f );
		glVertex3f( -headWidth, headLength, 0.0f );
		glVertex3f( headWidth, headLength, 0.0f );

		glColor3f(  0.0f, 0.0f, 1.0f ); // blue
		glVertex3f( 0.0f, 0.0f, axisLength );
		glVertex3f( -headWidth, 0.0f, headLength );
		glVertex3f( headWidth, 0.0f, headLength );
	glEnd();
	glPopAttrib();
}

/*
 * Animate() handles the animation and the redrawing of the
 *		graphics window contents.
 */
static void Animate(void)
{
	// Clear the rendering window
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    if (spinMode) {
		// Update the animation state
        HourOfDay += AnimateIncrement;  // bump by hour increment
        DayOfYear += (float)(AnimateIncrement/24.0);

        HourOfDay = HourOfDay - ((int)(HourOfDay/24))*24;
        DayOfYear = DayOfYear - ((int)(DayOfYear/365))*365;
	}
	
	// Clear the current matrix (Modelview)
    glLoadIdentity();

    // tramsform camera
    // ================
	// Back off eight units to be able to view from the origin.
    glTranslatef ( 0.0, 0.0, cameraDistance ); // -8.0 );
	// Rotate the plane of the elliptic
	// (rotate the model's plane about the x axis by fifteen degrees)
	// glRotatef( 15.0, 1.0, 0.0, 0.0 );
    glRotatef(cameraAngleX, 1, 0, 0);   // pitch
    glRotatef(cameraAngleY, 0, 1, 0);   // heading
    // ================

    // Draw the sun	-- as a yellow, wireframe sphere
	glColor3f( 1.0, 1.0, 0.0 );			
    glutWireSphere( 1.0, 15, 15 );
    XYZaxes();

    // Draw the Earth
	// First position it around the sun
	//		Use DayOfYear to determine its position
    glRotatef( (GLfloat)(360.0*DayOfYear/365.0), 0.0, 1.0, 0.0 );
    glTranslatef( 4.0, 0.0, 0.0 );
    glPushMatrix();						// Save matrix state
	// Second, rotate the earth on its axis.
	//		Use HourOfDay to determine its rotation.
	glRotatef( (GLfloat)(360.0*HourOfDay/24.0), 0.0, 1.0, 0.0 );
	// Third, draw the earth as a bluish wireframe sphere.
    glColor3f( 0.2, 0.2, 1.0 );
    glutWireSphere( 0.4, 10, 10);
    glPopMatrix();						// Restore matrix state

	// Draw the greenish moon.
	//	Use DayOfYear to control its rotation around the earth
   	glRotatef( (GLfloat)(360.0*12.0*DayOfYear/365.0), 0.0, 1.0, 0.0 );
    glTranslatef( 0.7, 0.0, 0.0 );
    glColor3f( 0.3, 0.7, 0.3 ); // greenish
    glutWireSphere( 0.1, 5, 5 );

    showFPS();  // add the FRAME RATE

	// Flush the pipeline, and swap the buffers
    glFlush();
    glutSwapBuffers();

	if ( singleStep ) {
		spinMode = GL_FALSE;
	}

	glutPostRedisplay();		// Request a re-draw for animation purposes

}

// Initialize OpenGL's rendering modes
void OpenGLInit(void)
{
    glShadeModel( GL_FLAT );
    glClearColor( 0.0, 0.0, 0.0, 0.0 );
    glClearDepth( 1.0 );
    glEnable( GL_DEPTH_TEST );
}

// ResizeWindow is called when the window is resized
static void ResizeWindow(int w, int h)
{
    float aspectRatio;
	h = (h == 0) ? 1 : h;
	w = (w == 0) ? 1 : w;
    screenWidth = w;
    screenHeight = h;

	glViewport( 0, 0, w, h );	// View port uses whole window
	aspectRatio = (float)w/(float)h;

	// Set up the projection view matrix (not very well!)
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity();
    gluPerspective( 60.0, aspectRatio, 1.0, 30.0 );

	// Select the Modelview matrix
    glMatrixMode( GL_MODELVIEW );
}

void mouseCB(int button, int state, int x, int y)
{
    mouseX = (float)x;
    mouseY = (float)y;

    if(button == GLUT_LEFT_BUTTON)
    {
        if(state == GLUT_DOWN)
        {
            mouseLeftDown = true;
        }
        else if(state == GLUT_UP)
            mouseLeftDown = false;
    }

    else if(button == GLUT_RIGHT_BUTTON)
    {
        if(state == GLUT_DOWN)
        {
            mouseRightDown = true;
        }
        else if(state == GLUT_UP)
            mouseRightDown = false;
    }
}


void mouseMotionCB(int x, int y)
{
    if(mouseLeftDown)
    {
        cameraAngleY += (x - mouseX);
        cameraAngleX += (y - mouseY);
        mouseX = (float)x;
        mouseY = (float)y;
    }
    if(mouseRightDown)
    {
        cameraDistance += (y - mouseY) * 0.2f;
        mouseY = (float)y;
    }
}


///////////////////////////////////////////////////////////////////////////////
// write 2d text using GLUT
// The projection matrix must be set to orthogonal before call this function.
///////////////////////////////////////////////////////////////////////////////
void drawString(const char *str, int x, int y, float color[4], void *font)
{
    glPushAttrib(GL_LIGHTING_BIT | GL_CURRENT_BIT); // lighting and color mask
    glDisable(GL_LIGHTING);     // need to disable lighting for proper text color
    glDisable(GL_TEXTURE_2D);

    glColor4fv(color);          // set text color
    glRasterPos2i(x, y);        // place text position

    // loop all characters in the string
    while(*str)
    {
        glutBitmapCharacter(font, *str);
        ++str;
    }

    glEnable(GL_TEXTURE_2D);
    glEnable(GL_LIGHTING);
    glPopAttrib();
}

///////////////////////////////////////////////////////////////////////////////
// display frame rates
///////////////////////////////////////////////////////////////////////////////
void showFPS()
{
    static Timer timer;
    static int count = 0;
    static std::stringstream ss;
    double elapsedTime;

    // backup current model-view matrix
    glPushMatrix();                     // save current modelview matrix
    glLoadIdentity();                   // reset modelview matrix

    // set to 2D orthogonal projection
    glMatrixMode(GL_PROJECTION);        // switch to projection matrix
    glPushMatrix();                     // save current projection matrix
    glLoadIdentity();                   // reset projection matrix
    gluOrtho2D(0, screenWidth, 0, screenHeight); // set to orthogonal projection

    float color[4] = {1, 1, 0, 1};

    // update fps every second
    elapsedTime = timer.getElapsedTime();
    if(elapsedTime < 1.0)
    {
        ++count;
    }
    else
    {
        ss.str("");
        ss << std::fixed << std::setprecision(1);
        ss << (count / elapsedTime) << " FPS" << ends; // update fps string
        ss << std::resetiosflags(std::ios_base::fixed | std::ios_base::floatfield);
        count = 0;                      // reset counter
        timer.start();                  // restart timer
    }
    drawString(ss.str().c_str(), screenWidth-85, screenHeight-14, color, font);

    // restore projection matrix
    glPopMatrix();                      // restore to previous projection matrix

    // restore modelview matrix
    glMatrixMode(GL_MODELVIEW);         // switch to modelview matrix
    glPopMatrix();                      // restore to previous modelview matrix
}

// Main routine
// Set up OpenGL, hook up callbacks, and start the main loop
int main( int argc, char** argv )
{
	// Need to double buffer for animation
	glutInit(&argc,argv);
    glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH );

	// Create and position the graphics window
    glutInitWindowPosition( 0, 0 );
    glutInitWindowSize( screenWidth, screenHeight );
    glutCreateWindow( "Solar System Demo 2" );

	// Initialize OpenGL.
    OpenGLInit();

	// Set up callback functions for key presses
	glutKeyboardFunc( KeyPressFunc );
	glutSpecialFunc( SpecialKeyFunc );

	// Set up the callback function for resizing windows
    glutReshapeFunc( ResizeWindow );

	// Callback for graphics image redrawing
    glutDisplayFunc( Animate );

    // call backs for mouse actions
    glutMouseFunc(mouseCB);
    glutMotionFunc(mouseMotionCB);

    Key_R();    // set-reset intitial values

    // show keyboard
    Key_Help();

	// Start the main loop.  glutMainLoop never returns.
	glutMainLoop(  );

    return(0);			// Compiler requires this to be here. (Never reached)
}
