/****************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the examples of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:BSD$
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QtGui>
#include <QMessageBox>
#include <QSettings>

#include "window.h"

QString company = "GTools";
QString appname = "QtFind";
QString title = "Qt Find Files";

// ===============================================================
// contents of the about box, if there was one
QString about = "QtFind v.0.0.1\n\n"
                "A grapical user interface\nfor searching files for a string.\n"
                "GNU General Public License - GPL2 (or later)\n\n"
                "Commenced Geoff R. McLane 2011\n";
// ===============================================================

QSettings settings(company,appname); // establish persistent settings
// WIN32: In registry - HKEY_CURRENT_USER/Software/TerraGear/<name>
// unix: ???

QString m_fileMask;
QString m_fmPath = "find/path";
QString m_findText;
QString m_ftPath = "find/text";
QString m_findDir;
QString m_fdPath = "find/dir";
bool m_check_recur;
QString m_crPath = "check/recursive";

// program variables
static int m_total_dirs;
static int m_total_found;
static int m_file_count;
static bool m_user_cancel;
static QString base_info;

// given millisecond, return appropriate (nice) string
static QString getElapTimeStg(int ms)
{
    QString fmt = "";
    if (ms < 1000) {
        fmt.sprintf("%d ms", ms);
    } else {
        int secs = ms / 1000;
        ms = ms % 1000;
        if (secs < 60) {
            if (ms)
                fmt.sprintf("%d.%03d secs", secs, ms);
            else
                fmt.sprintf("%d secs", secs);
        } else {
            int mins = secs / 60;
            secs = secs % 60;
            if (mins < 60) {
                if (ms)
                    fmt.sprintf("%d:%02d.%03d mins", mins, secs, ms);
                else {
                    if (secs)
                        fmt.sprintf("%d:%02d mins", mins, secs);
                    else
                        fmt.sprintf("%d mins", mins);
                }
            } else {
                int hrs = mins / 60;
                mins = mins % 60;
                if (ms)
                    fmt.sprintf("%d:%02d:%02d.%03d hrs", hrs, mins, secs, ms);
                else {
                    if (secs)
                        fmt.sprintf("%d:%02d:%02d hrs", hrs, mins, secs);
                    else {
                        if (mins)
                            fmt.sprintf("%d:%02d hrs", hrs, mins);
                        else
                            fmt.sprintf("%d hrs", hrs);
                    }
                }
            }
        }
    }
    return fmt;
}

static void addtoComboBox(QComboBox *comboBox,QString item)
{
    if (item.size() == 0)
        return;
    if (comboBox->findText(item) == -1)
        comboBox->addItem(item);
    comboBox->setCurrentIndex(comboBox->findText(item));
}

static void updateComboBox(QComboBox *comboBox)
{
    if (comboBox->findText(comboBox->currentText()) == -1)
        comboBox->addItem(comboBox->currentText());
}

// ==============================================
// Window implementation
// ==============================================
Window::Window(QWidget *parent)
    : QDialog(parent)
{
    QString tmp;

    pPD = 0;
    currentRT = new QTime;
    m_user_cancel = false;

    browseButton = createButton(tr("&Browse..."), SLOT(browse()));
    findButton = createButton(tr("&Find"), SLOT(find()));
    quitButton = createButton(tr("&Quit"), SLOT(quit()));

    recursive = new QCheckBox(tr("Recursive")); // QCheckBox

    // get any previous settings
    tmp = settings.value(m_fmPath).toString();
    m_fileMask = tmp;
    if (m_fileMask.size() == 0)
        m_fileMask = "*";

    tmp = settings.value(m_ftPath).toString();
    m_findText = tmp;

    tmp = settings.value(m_fdPath).toString();
    m_findDir = tmp;
    if (m_findDir.size()==0)
        m_findDir = QDir::currentPath();
    m_check_recur = settings.value(m_crPath).toBool();

    fileComboBox = createComboBox();
    textComboBox = createComboBox();
    directoryComboBox = createComboBox();

    fileLabel = new QLabel(tr("Named:"));
    textLabel = new QLabel(tr("Containing text:"));
    directoryLabel = new QLabel(tr("In directory:"));
    filesFoundLabel = new QLabel;

    createFilesTable();

    QHBoxLayout *buttonsLayout = new QHBoxLayout;
    buttonsLayout->addStretch();
    buttonsLayout->addWidget(recursive);
    buttonsLayout->addWidget(quitButton);
    buttonsLayout->addWidget(findButton);

    QGridLayout *mainLayout = new QGridLayout;
    mainLayout->addWidget(fileLabel, 0, 0);
    mainLayout->addWidget(fileComboBox, 0, 1, 1, 2);
    mainLayout->addWidget(textLabel, 1, 0);
    mainLayout->addWidget(textComboBox, 1, 1, 1, 2);
    mainLayout->addWidget(directoryLabel, 2, 0);
    mainLayout->addWidget(directoryComboBox, 2, 1);
    mainLayout->addWidget(browseButton, 2, 2);
    mainLayout->addWidget(filesTable, 3, 0, 1, 3);
    mainLayout->addWidget(filesFoundLabel, 4, 0, 1, 3);
    mainLayout->addLayout(buttonsLayout, 5, 0, 1, 3);
    setLayout(mainLayout);

    // add previous items
    // need a way to store and retrieve a LIST... ***TBD***
    addtoComboBox(fileComboBox,     m_fileMask);
    addtoComboBox(textComboBox,     m_findText);
    addtoComboBox(directoryComboBox,m_findDir );
    recursive->setCheckState(m_check_recur ? Qt::Checked : Qt::Unchecked);

    //setWindowTitle(tr("Find Files"));
    setWindowTitle(title);
    resize(700, 300);
}

Window::~Window()
{
    if (pPD) {
        if (!pPD->isHidden())
            pPD->hide();
        delete pPD;
    }
}

/*
    Utility function that recursively searches, if desired, for files per filters.
*/
QStringList Window::findFiles(const QString &startDir, QStringList filters, bool recurse)
{
    QStringList names;
    QDir dir(startDir);
    QString tmp;
    int secs;
    // 1: Get the files matching the filter
    foreach (QString file, dir.entryList(filters, QDir::Files))
        names += startDir + "/" + file;

    m_total_dirs++;
    if (names.size() && pPD) {
        m_total_found += names.size();
        secs = currentRT->elapsed() / 1000;
        tmp.sprintf("\nfound %d...",m_total_found);
        tmp += " in "+getElapTimeStg(secs * 1000);
        pPD->setLabelText(base_info+tmp);
        qApp->processEvents();
        if (m_user_cancel || pPD->wasCanceled() ) {
            m_user_cancel = true;
            return names;
        }
        // pPD->repaint();
    }
    if (recurse) {
        // 2: If recursive, get ALL directories, and try for more files
        foreach (QString subdir, dir.entryList(QDir::AllDirs | QDir::NoDotAndDotDot)) {
            names += findFiles(startDir + "/" + subdir, filters, recurse);
            if (pPD && (m_user_cancel || pPD->wasCanceled())) {
                m_user_cancel = true;
                break;
            }
        }
    }
    return names;
}

void Window::browse()
{
    QString directory = QFileDialog::getExistingDirectory(this,
                               tr("Find Files"), QDir::currentPath());

    if (!directory.isEmpty()) {
        addtoComboBox(directoryComboBox, directory);
    }
}

void Window::quit()
{
    close();
}

QStringList Window::findInFiles(const QStringList &files, const QString &text)
{
    QTime rt;
    int i, secs;
    QString info;

    if (pPD) {
        pPD->setCancelButtonText(tr("&Cancel"));
        pPD->setRange(0, files.size());
        pPD->setWindowTitle(tr("Finding IN Files"));
    } else {
        QProgressDialog progressDialog(this);
        pPD = &progressDialog;
    }
    QStringList foundFiles;
    rt.start();

    for (i = 0; i < files.size(); ++i) {
        info.sprintf("Searching %d of %d... found %d...", (i+1), files.size(), foundFiles.size());
        secs = rt.elapsed() / 1000;
        info += "\nin "+getElapTimeStg(secs * 1000);
        secs = currentRT->elapsed() / 1000;
        info += ", tot "+getElapTimeStg(secs * 1000);
        if (pPD) {
            pPD->setValue(i);
            pPD->setLabelText(info);
        }
        qApp->processEvents();

        if (pPD && (m_user_cancel || pPD->wasCanceled())) {
            m_user_cancel = true;
            break;
        }

        QFile file(currentDir.absoluteFilePath(files[i]));

        if (file.open(QIODevice::ReadOnly)) {
            QString line;
            QTextStream in(&file);
            while (!in.atEnd()) {
                if (pPD && (m_user_cancel || pPD->wasCanceled())) {
                    m_user_cancel = true;
                    break;
                }
                line = in.readLine();
                if (line.contains(text)) {
                    foundFiles << files[i];
                    break;
                }
            }
        }
    }
    if (files.size() && !m_user_cancel) {
        info.sprintf("Searched %d of %d... found %d", (i+1), files.size(), foundFiles.size());
        secs = rt.elapsed() / 1000;
        info += "... in "+getElapTimeStg(secs * 1000);
        if (pPD) {
            pPD->setValue(i);
            pPD->setLabelText(info);
        }
        qApp->processEvents();
        //rt.restart();
        //while (rt.elapsed() < 5000)
        //    i++;
    }
    return foundFiles;
}


void Window::find()
{
    bool recur = recursive->isChecked();
    QString tmp;
    QString info;

    m_total_dirs = 0;
    m_total_found = 0;
    m_file_count = 0;
    m_user_cancel = false;

    currentRT->start();

    filesTable->setRowCount(0);

    QString fileName = fileComboBox->currentText();
    QString text = textComboBox->currentText();
    QString path = directoryComboBox->currentText();

    if (!pPD) {
        pPD = new QProgressDialog(this);
    }
    if (pPD) {
        pPD->setAutoClose(false);
        pPD->setAutoReset(false);
        pPD->reset();
        pPD->setCancelButtonText(tr("&Cancel"));
        pPD->setRange(0, 100);
        pPD->setWindowTitle(tr("Finding Files"));
        pPD->setValue(0);
        info = "In path ["+path+"] finding ["+fileName+"] files";
        if (recur)
            info += ", recursively...";
        base_info = info;
        pPD->setLabelText(info);
        pPD->setAutoClose(false);
        pPD->setAutoReset(false);
        pPD->show();
        pPD->repaint();
    }

    updateComboBox(fileComboBox);
    updateComboBox(textComboBox);
    updateComboBox(directoryComboBox);

    currentDir = QDir(path);
    if (!currentDir.exists()) {
        if (pPD)
            pPD->hide();
        info = "ERROR: Directory ["+path+"] does NOT exist!";
        QMessageBox::information(this,"INVALID DIRECTORY",info); // Yeek
        return;
    }

    QStringList files;
    if (fileName.isEmpty())
        fileName = "*";

    // this works well, and is quite quick and efficient ;=))
    files = findFiles(path, QStringList() << fileName, recur);
    if (pPD && (m_user_cancel || pPD->wasCanceled())) {
        m_user_cancel = true;
        pPD->hide();
        return;
    }
    m_file_count = files.size();
    if (files.isEmpty()) {
        if (pPD)
            pPD->hide();
        info = "Using the path ["+path+"]\nand the file filter ["+fileName+"]";
        if (recur) {
            info += "\nEven thought the recursive switch is on";
            tmp.sprintf("\nand searched %d directories.", m_total_dirs);
            info += tmp;
        }
        info += "\n\nFOUND NO MATCHING FILES!";
        showFiles(files);
        QMessageBox::information(this,"NO FILES FOUND!",info); // Yeek
        return;
    }

    if (m_user_cancel) {
        if (pPD)
            pPD->hide();
        return; // all done
    }

    // had a successful find, so update presistance
    // ============================================
    settings.setValue(m_fmPath, fileName);
    settings.setValue(m_ftPath, text    );
    settings.setValue(m_fdPath, path    );
    if (recur != m_check_recur ) {
        m_check_recur = recur;
        settings.setValue(m_crPath,m_check_recur);
    }
    // ============================================

    if (!text.isEmpty()) {
        // search for text in file, using a QProgressDialog
        // ================================================
        files = findInFiles(files, text);
        // ================================================
        if (m_user_cancel) {
            if (pPD)
                pPD->hide();
            return;
        }
        if (files.isEmpty()) {
            if (pPD)
                pPD->hide();
            info = "Using the path ["+path+"]\nand the file filter ["+fileName+"]";
            if (recur) {
                tmp.sprintf("\nwith recursive on searched %d directories.", m_total_dirs);
                info += tmp;
            }
            tmp.sprintf("\nFound %d matching files.", m_file_count);
            info += tmp;
            info += "\nBut NONE contain the text ["+text+"]!";
            showFiles(files);
            QMessageBox::information(this,"NO FILES FOUND!",info); // Yeek
            return;
        }
    }

    showFiles(files);
    if (pPD)
        pPD->hide();
}


void Window::showFiles(const QStringList &files)
{
    for (int i = 0; i < files.size(); ++i) {
        QFile file(currentDir.absoluteFilePath(files[i]));
        qint64 size = QFileInfo(file).size();

        // Column 1
        QTableWidgetItem *fileNameItem = new QTableWidgetItem(files[i]);
        fileNameItem->setFlags(fileNameItem->flags() ^ Qt::ItemIsEditable);

        // Column 2
        QTableWidgetItem *sizeItem = new QTableWidgetItem(tr("%1 KB")
                                             .arg(int((size + 1023) / 1024)));
        sizeItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
        sizeItem->setFlags(sizeItem->flags() ^ Qt::ItemIsEditable);

        // create a NEW row
        int row = filesTable->rowCount();
        filesTable->insertRow(row);
        // set column text
        filesTable->setItem(row, 0, fileNameItem);
        filesTable->setItem(row, 1, sizeItem);
    }

    // add summary
    int found_cnt = files.size();
    QString info = tr("%1 file(s) found").arg(found_cnt);
    if (found_cnt)
        info += " (Double click on a file to open it)";
    filesFoundLabel->setText(info);

}

QPushButton *Window::createButton(const QString &text, const char *member)
{
    QPushButton *button = new QPushButton(text);
    connect(button, SIGNAL(clicked()), this, member);
    return button;
}

QComboBox *Window::createComboBox(const QString &text)
{
    QComboBox *comboBox = new QComboBox;
    comboBox->setEditable(true);
    comboBox->addItem(text);
    comboBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
    return comboBox;
}

void Window::createFilesTable()
{
    filesTable = new QTableWidget(0, 2);
    filesTable->setSelectionBehavior(QAbstractItemView::SelectRows);

    QStringList labels;
    labels << tr("File Name") << tr("Size");
    filesTable->setHorizontalHeaderLabels(labels);
    filesTable->horizontalHeader()->setResizeMode(0, QHeaderView::Stretch);
    filesTable->verticalHeader()->hide();
    filesTable->setShowGrid(false);

    connect(filesTable, SIGNAL(cellActivated(int,int)),
            this, SLOT(openFileOfItem(int,int)));
}

void Window::openFileOfItem(int row, int /* column */)
{
    QTableWidgetItem *item = filesTable->item(row, 0);

    QDesktopServices::openUrl(QUrl::fromLocalFile(currentDir.absoluteFilePath(item->text())));
}


